function [ k,equil_mass ] = Tube_Diff_Fit_1d()
%Input geometry and time till midpoint is half-full to get diff coeff.
%   The program works by first simulating a dirichlet condition with unity
%   diffusion.  Once 1/2 of the equilibrium total mass is achieved, a
%   scaling symmetry is used to adjust the measured time to the user-input
%   time.  The scaling symmetry then determines the diffusion coefficient.
%   Note that the dirichlet condition is taken as unity.  If the tfinal
%   time is not sufficient to reach the half-way mark, then the program
%   returns k = NaN and prints the last u_avg and it's difference from the
%   half-way mark.

r = input('r of S^r_d:');
d = input('d of S^r_d:');
Radn = input('Number of mesh intervals along sphere height:');
Cyln = input('Number of mesh intervals along cylinder height:');
Rad  = input('Radius:');
Ht   = input('Height:');
thalf    = input('Time at which half total mass:');
tstep      = input('Time Increment:');
tfinal = input('Guess time before which this tube with unity diffusion will have attained half its equilibrium total mass:');
cdir       = 1;

%Construct the mesh on [-r,H] interval
orig_delta = [-1*Rad:Rad/Radn:0 (Ht-0)/Cyln:(Ht-0)/Cyln:Ht]; %Put knots at the 1/Radn and 1/Cyln spacing. 
[extdelta,n] = extpar(orig_delta,r,d);                       %Guaranteed 1-knot at 0.

%Define 0 initital data
cinitial = zeros(1,n);

%Compute the total equilibrium mass
cfinal = zeros(1,n)+1; %The basis splines form a partition of unity
equil_mass = 2*pi*Rad*intspl(d,extdelta,cfinal,-Rad,Ht);

%Run the Dirichlet diffusion for unity coefficient:
%Copied from Master_Sim_1d
S1 = Assemble_Time(orig_delta,r,d,@(x) 1);                                       %The time component coefficient is 
                                                                                 %\equiv 1 for us                                                                                 
S2 = Assemble_Space(orig_delta,r,d,@(x) ((Rad^2-x^2)/(Rad^2)).*(x<=0) + (x>0));

A = S1(1:n-1,1:n-1);
B = S2(1:n-1,1:n-1);
f = cdir*S2(1:n-1,n);

%0 = Ac' + Bc + f
DE = @(t,y) A^(-1)*(-B*y-f); %ODE equation for input in ODE45
y0 = cinitial(1,1:n-1)'; %Vector of initial data
[tout,yout] = ode45(DE,0:tstep:tfinal,y0); 

%Augment yout to include now the constant cn = cdir term.
yout = [yout cdir+zeros(size(yout,1),1)];

%Build the averages.
tcount = size(tout,1);
u_avg = zeros(tcount,1);
for i=1:tcount
    u_avg(i,1) = 2*pi*Rad*intspl(d,extdelta,yout(i,:),-Rad,Ht); %\int ud\sigma = \int^(2pi)_(\theta=0)\int^H_(-r) urdzd\theta
end

if u_avg(tcount,1)<.5*equil_mass
    k = NaN;
    [u_avg(tcount,1) .5*equil_mass - u_avg(tcount,1)] %Print the final avg and its difference from the 
                                                      %.5*equil_mass mark.
else 
    %Find where u_avg goes from <.5*equil_mass --> >= .5*equil_mass
    ktr = 1;
    while u_avg(ktr,1)<.5*equil_mass
        ktr = ktr + 1;
    end
    
%Compute next the linear interpolation weight and the time with
%unity diffusion coefficient tmeas till reach .5*equi_mass.
%This derived from u_avg(ktr-1,1)<.5*equil_mass<=u_avg(ktr,1)
%(1-s)u_avg(ktr-1,1)+su_avg(ktr,1) = .5*equil_mass
s = (.5*equil_mass - u_avg(ktr-1,1))/(u_avg(ktr,1) - u_avg(ktr-1,1));
tmeas = (1-s)*tout(ktr-1,1)+s*tout(ktr,1); 

%Rescale the PDE using that v(x;t) = u(x;\lambda t) has v_t - \lambda\Delta v = 0
%Thus if v(x;thalf) = u(x;\lambda thalf) = u(x;tmeas)
k = tmeas/thalf;
end
end

